﻿using System;
using System.ComponentModel;
using System.Linq;
using System.Reactive.Linq;
using System.Threading.Tasks;
using System.Windows;
using Azyobuzi.Twikoto2.Models;
using Azyobuzi.Twikoto2.Models.Twitter;
using Livet;
using Livet.Command;
using Livet.Messaging;
using Livet.Messaging.Window;

namespace Azyobuzi.Twikoto2.ViewModels
{
    public class MainWindowViewModel : ViewModel
    {
        /*コマンド、プロパティの定義にはそれぞれ 
         * 
         *  ldcom   : DelegateCommand(パラメータ無)
         *  ldcomn  : DelegateCommand(パラメータ無・CanExecute無)
         *  ldcomp  : DelegateCommand(型パラメータ有)
         *  ldcompn : DelegateCommand(型パラメータ有・CanExecute無)
         *  lprop   : 変更通知プロパティ
         *  
         * を使用してください。
         */

        /*ViewModelからViewを操作したい場合は、
         * Messengerプロパティからメッセージ(各種InteractionMessage)を発信してください。
         */

        /*
         * UIDispatcherを操作する場合は、DispatcherHelperのメソッドを操作してください。
         * UIDispatcher自体はApp.xaml.csでインスタンスを確保してあります。
         */

        /*
         * Modelからの変更通知などの各種イベントをそのままViewModelで購読する事はメモリリークの
         * 原因となりやすく推奨できません。ViewModelHelperの各静的メソッドの利用を検討してください。
         */

        public MainWindowViewModel()
        {
            ViewModelHelper.BindNotifyChanged(this.model, this, (sender, e) =>
            {
                switch (e.PropertyName)
                {
                    case "Timeline":
                        this.RaisePropertyChanged(e.PropertyName);
                        break;
                }
            });

            ViewModelHelper.BindNotification(
                this.model.RequestPinCodeEvent,
                this,
                this.model_RequestPinCode);

            this.QueryTypes = new DispatcherCollection<QueryViewModel>(
                new QueryViewModel[]
                {
                    new QueryViewModel() { Type = TimelineTypes.None, Name = "None" },
                    new QueryViewModel() { Type = TimelineTypes.Home, Name = "Home" },
                    new QueryViewModel() { Type = TimelineTypes.Mentions, Name = "Mentions" }
                },
                DispatcherHelper.UIDispatcher);
            this.SelectedQuery = this.QueryTypes[0];
        }

        private Model model = new Model();

        public Settings Settings
        {
            get
            {
                return this.model.Settings;
            }
        }

        private void GetLists()
        {
            this.model.GetListNames()
                .Select(_ => new QueryViewModel()
                {
                    Type = TimelineTypes.List,
                    Name = "List : " + _,
                    Parameter = _
                })
                .Subscribe(
                    this.QueryTypes.Add,
                    ex => DispatcherHelper.BeginInvoke(() =>
                        this.Messenger.Raise(new InformationMessage(
                            ex.ToString(),
                            "リスト取得失敗",
                            MessageBoxImage.Error,
                            "ShowInfo")))
                );
        }

        #region LoadedCommand
        DelegateCommand _LoadedCommand;

        public DelegateCommand LoadedCommand
        {
            get
            {
                if (_LoadedCommand == null)
                    _LoadedCommand = new DelegateCommand(Loaded);
                return _LoadedCommand;
            }
        }

        private void Loaded()
        {
            this.model.Authorize();

            if (this.model.IsAuthorized)
                this.GetLists();
        }
        #endregion

        private void model_RequestPinCode(object sender, RequestPinCodeEventArgs e)
        {
            DispatcherHelper.BeginInvoke(() =>
            {
                this.Authorizing = true;
                var vm = new InputPinWindowViewModel()
                {
                    AuthorizeUri = new Uri(e.AuthorizeUri),
                    RequestToken = e.RequestToken
                };
                vm.PropertyChanged += this.InputPinWindowViewModel_PropertyChanged;
                this.Messenger.Raise(new TransitionMessage(vm, "GetPinCode"));
            });
        }

        private void InputPinWindowViewModel_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            var vm = (InputPinWindowViewModel)sender;
            if (e.PropertyName == "Closed" && vm.Closed)
            {
                vm.PropertyChanged -= this.InputPinWindowViewModel_PropertyChanged;
                this.model.EndAuthorize(vm.RequestToken, vm.PinCode)
                    .ObserveOnDispatcher()
                    .Finally(() => this.Authorizing = false)
                    .Subscribe(
                        _ => this.GetLists(),
                        ex =>
                        {
                            this.Messenger.Raise(new InformationMessage(
                             ex.ToString(),
                             "認証失敗",
                             MessageBoxImage.Error,
                             "ShowInfo"));
                            this.Messenger.Raise(new WindowActionMessage("ChangeWindowState", WindowAction.Close));
                        }
                    );

            }
        }

        bool _Authorizing;

        public bool Authorizing
        {
            get
            { return _Authorizing; }
            set
            {
                if (_Authorizing == value)
                    return;
                _Authorizing = value;
                RaisePropertyChanged("Authorizing");
            }
        }

        #region ClosingCommand
        DelegateCommand _ClosingCommand;

        public DelegateCommand ClosingCommand
        {
            get
            {
                if (_ClosingCommand == null)
                    _ClosingCommand = new DelegateCommand(Closing);
                return _ClosingCommand;
            }
        }

        private void Closing()
        {
            this.Settings.Save();
        }
        #endregion

        string _StatusText = "";

        public string StatusText
        {
            get
            { return _StatusText; }
            set
            {
                if (_StatusText == value)
                    return;
                _StatusText = value;
                RaisePropertyChanged("StatusText");
                RaisePropertyChanged(() => this.OverStatusText);
            }
        }

        public bool OverStatusText
        {
            get
            {
                return StatusText.Trim().Length > 140;
            }
        }

        string _InReplyToId = "";

        public string InReplyToId
        {
            get
            { return _InReplyToId; }
            set
            {
                if (_InReplyToId == value)
                    return;
                _InReplyToId = value.Trim();//先に格納してエラーを把握する
                if (!_InReplyToId.All(_ => "0123456789".Contains(_)))
                    throw new ArgumentException("ReplyToIdは正の整数でなければいけません。");
                RaisePropertyChanged("InReplyToId");
            }
        }

        bool _Posting;

        public bool Posting
        {
            get
            { return _Posting; }
            private set
            {
                if (_Posting == value)
                    return;
                _Posting = value;
                RaisePropertyChanged("Posting");
            }
        }

        #region PostCommand
        DelegateCommand _PostCommand;

        public DelegateCommand PostCommand
        {
            get
            {
                if (_PostCommand == null)
                    _PostCommand = new DelegateCommand(Post, CanPost);
                return _PostCommand;
            }
        }

        private bool CanPost()
        {
            return !string.IsNullOrWhiteSpace(this.StatusText) &&
                !this.OverStatusText &&
                (string.IsNullOrEmpty(this.InReplyToId) || this.InReplyToId.All(_ => "0123456789".Contains(_)));
        }

        private void Post()
        {
            this.Posting = true;
            this.model.UpdateStatus(this.StatusText, this.InReplyToId)
                .ObserveOnDispatcher()
                .Finally(() => this.Posting = false)
                .Subscribe(
                    _ =>
                    {
                        this.StatusText = "";
                        this.InReplyToId = "";
                    },
                    ex => this.Messenger.Raise(new InformationMessage(
                        ex.ToString(),
                        "投稿失敗",
                        MessageBoxImage.Error,
                        "ShowInfo")));

        }
        #endregion

        public DispatcherCollection<QueryViewModel> QueryTypes { private set; get; }

        QueryViewModel _SelectedQuery;

        public QueryViewModel SelectedQuery
        {
            get
            { return _SelectedQuery; }
            set
            {
                if (_SelectedQuery == value)
                    return;
                _SelectedQuery = value;
                RaisePropertyChanged("SelectedQuery");
                RaisePropertyChanged(() => this.TimelineTypeIsNone);
                this.model.GetTimeline(value.Model)
                    .ContinueWith(this.GetTimelineCallback);
            }
        }

        private void GetTimelineCallback(Task t)
        {
            if (t.Exception != null)
            {
                DispatcherHelper.BeginInvoke(() =>
                    this.Messenger.Raise(new InformationMessage(
                        t.Exception.ToString(),
                        "タイムライン取得失敗",
                        MessageBoxImage.Error,
                        "ShowInfo")));
            }
        }

        public DispatcherCollection<Status> Timeline
        {
            get
            {
                return this.model.Timeline;
            }
        }

        public bool TimelineTypeIsNone
        {
            get
            {
                return this.SelectedQuery == null ||
                    this.SelectedQuery.Type == TimelineTypes.None;
            }
        }
    }
}
